<?php
include('includes/header.php');

// Ensure the database is initialized
initializeDatabase($db);

// Handle form submission for adding new URLs
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['add_url'])) {
    $url = sanitize($_POST['url']);
    
    // Add the new URL
    $stmt = $db->prepare("INSERT INTO ads2_images (ads2_id, url) VALUES (1, :url)");
    $stmt->bindValue(':url', $url, SQLITE3_TEXT);
    
    if ($stmt->execute()) {
        // If a URL is manually added, automatically disable TMDB API
        $db->exec("UPDATE settings SET tmdb_api_enabled = 0 WHERE id = 1");
        echo "<div class='alert alert-success'>URL added successfully. TMDB API has been automatically disabled.</div>";
    } else {
        echo "<div class='alert alert-danger'>Failed to add URL: " . $db->lastErrorMsg() . "</div>";
    }
}

// Handle deletion of URLs
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['delete_url'])) {
    $id = (int)$_POST['id'];
    $stmt = $db->prepare("DELETE FROM ads2_images WHERE id = :id");
    $stmt->bindValue(':id', $id, SQLITE3_INTEGER);
    
    if ($stmt->execute()) {
        // Check if there are remaining manual URLs
        $count = $db->querySingle("SELECT COUNT(*) FROM ads2_images WHERE ads2_id = 1");
        
        if ($count == 0) {
            // Enable TMDB if no manual URLs remain and the user had it enabled previously
            $tmdbState = $db->querySingle("SELECT tmdb_api_enabled FROM settings WHERE id=1");
            // Reactivate only if it was previously enabled
            if ($tmdbState) {
                $db->exec("UPDATE settings SET tmdb_api_enabled = 1 WHERE id = 1");
                echo "<div class='alert alert-success'>URL deleted successfully. No manual images left - TMDB API has been automatically re-enabled.</div>";
            } else {
                echo "<div class='alert alert-success'>URL deleted successfully. No manual images left.</div>";
            }
        } else {
            echo "<div class='alert alert-success'>URL deleted successfully.</div>";
        }
    } else {
        echo "<div class='alert alert-danger'>Failed to delete URL: " . $db->lastErrorMsg() . "</div>";
    }
}

// Handle form submission for updating ads table
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['update_ads'])) {
    $text1 = sanitize($_POST['text1']);

    // Update ads table
    $stmt1 = $db->prepare("INSERT OR REPLACE INTO ads (id, text) VALUES (1, :text)");
    $stmt1->bindValue(':text', $text1, SQLITE3_TEXT);
    if ($stmt1->execute()) {
        echo "<div class='alert alert-success'>Ads table updated successfully.</div>";
    } else {
        echo "<div class='alert alert-danger'>Failed to update ads table: " . $db->lastErrorMsg() . "</div>";
    }
}

// Handle form submission for updating ads2 table
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['update_ads2'])) {
    $text2 = sanitize($_POST['text2']);

    // Update ads2 table
    $stmt2 = $db->prepare("INSERT OR REPLACE INTO ads2 (id, text) VALUES (1, :text)");
    $stmt2->bindValue(':text', $text2, SQLITE3_TEXT);
    if ($stmt2->execute()) {
        echo "<div class='alert alert-success'>Ads2 table updated successfully.</div>";
    } else {
        echo "<div class='alert alert-danger'>Failed to update ads2 table: " . $db->lastErrorMsg() . "</div>";
    }
}

// Handle form submission for updating TMDB API state
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['toggle_tmdb'])) {
    $tmdb_api_enabled = (int)$_POST['tmdb_api_enabled'];
    
    // Check if there are manual images
    $count = $db->querySingle("SELECT COUNT(*) FROM ads2_images WHERE ads2_id = 1");
    
    if ($tmdb_api_enabled == 1 && $count > 0) {
        // If they try to enable TMDB but there are manual images, show warning
        echo "<div class='alert alert-warning'>Cannot enable TMDB API while manual images exist. Please delete all manual images first.</div>";
    } else {
        // Update TMDB API state
        $stmt3 = $db->prepare("UPDATE settings SET tmdb_api_enabled = :tmdb_api_enabled WHERE id = 1");
        $stmt3->bindValue(':tmdb_api_enabled', $tmdb_api_enabled, SQLITE3_INTEGER);
        if ($stmt3->execute()) {
            echo "<div class='alert alert-success'>TMDB API state updated successfully.</div>";
        } else {
            echo "<div class='alert alert-danger'>Failed to update TMDB API state: " . $db->lastErrorMsg() . "</div>";
        }
    }
}

// Fetch existing data for display in the form
$res1 = $db->query("SELECT * FROM ads WHERE id=1");
$rowU1 = $res1 ? $res1->fetchArray(SQLITE3_ASSOC) : ['text' => ''];

$res2 = $db->query("SELECT * FROM ads2 WHERE id=1");
$rowU2 = $res2 ? $res2->fetchArray(SQLITE3_ASSOC) : ['text' => ''];

$res3 = $db->query("SELECT * FROM ads2_images WHERE ads2_id=1");
$ads2Urls = [];
while ($row = $res3->fetchArray(SQLITE3_ASSOC)) {
    $ads2Urls[] = $row;
}

// Ensure TMDB API state is initialized
$tmdbApiState = $db->querySingle("SELECT tmdb_api_enabled FROM settings WHERE id=1");
if ($tmdbApiState === null) {
    $db->exec("INSERT INTO settings (id, tmdb_api_enabled) VALUES (1, 1)");
    $tmdbApiState = 1;
}

// Check if manual images exist
$manualImagesExist = count($ads2Urls) > 0;
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Ads Management Dashboard</title>

    <style>
        /* Custom styles */
        body {
            background-color: #000000;
            background-image: radial-gradient(circle, rgba(0, 153, 255, 0.3) 1px, transparent 1px),
                              radial-gradient(circle, rgba(255, 0, 255, 0.3) 1px, transparent 1px),
                              radial-gradient(circle, rgba(255, 255, 0, 0.3) 1px, transparent 1px),
                              radial-gradient(circle, rgba(0, 255, 0, 0.3) 1px, transparent 1px),
                              radial-gradient(circle, rgba(255, 0, 0, 0.3) 1px, transparent 1px);
            background-size: 50px 50px;
            background-position: 0 0, 25px 25px, 12px 12px, 38px 38px, 5px 40px;
            color: #ffffff;
            font-family: Arial, sans-serif;
        }
        
        .dashboard-header {
            background-color: #007bff;
            color: white;
            padding: 15px 0;
            margin-bottom: 30px;
            border-radius: 8px;
        }
        
        .card {
            background-color: #222222;
            border: 1px solid #444444;
            border-radius: 8px;
            margin-bottom: 20px;
        }
        
        .card-header {
            background-color: #2c3e50;
            color: white;
            padding: 15px;
            border-bottom: 1px solid #444444;
            border-radius: 8px 8px 0 0;
        }
        
        .card-body {
            padding: 20px;
        }
        
        .text-controls-panel {
            background-color: #222222;
            border-radius: 8px;
            padding: 20px;
            margin-bottom: 30px;
            color: white;
        }
        
        .control-box {
            padding: 15px;
            border: 1px solid #444444;
            border-radius: 5px;
            margin-bottom: 15px;
            background-color: #333333;
        }
        
        .control-box h4 {
            margin-top: 0;
            color: #ffffff;
            font-size: 1.1rem;
            border-bottom: 2px solid #444444;
            padding-bottom: 8px;
        }
        
        .api-settings-panel {
            background-color: #222222;
            border-radius: 8px;
            padding: 20px;
            margin-bottom: 30px;
            color: white;
        }
        
        .url-manager-panel {
            background-color: #222222;
            border-radius: 8px;
            padding: 20px;
            margin-bottom: 30px;
            color: white;
        }
        
        .gallery-container {
            background-color: #222222;
            border-radius: 8px;
            padding: 0;
            margin-bottom: 30px;
            color: white;
        }
        
        .gallery-title {
            color: #ffffff;
            margin-bottom: 20px;
        }
        
        .image-showcase {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(160px, 1fr));
            gap: 15px;
            padding: 15px;
        }
        
        .image-item {
            position: relative;
            border-radius: 8px;
            overflow: hidden;
            border: 1px solid #444444;
        }
        
        .image-item img {
            width: 100%;
            height: 160px;
            object-fit: cover;
        }
        
        .delete-overlay {
            position: absolute;
            bottom: 0;
            left: 0;
            right: 0;
            padding: 8px;
            text-align: center;
            background-color: rgba(0, 0, 0, 0.5);
        }
        
        .btn-custom-blue {
            background-color: #3498db;
            border-color: #2980b9;
            color: white;
        }
        
        .btn-custom-darkblue {
            background-color: #2c3e50;
            border-color: #1a252f;
            color: white;
        }
        
        .form-control {
            background-color: #333333;
            border: 1px solid #555555;
            color: white;
        }
        
        .form-control:focus {
            background-color: #444444;
            color: white;
            border-color: #007bff;
            box-shadow: 0 0 0 0.2rem rgba(0, 123, 255, 0.25);
        }
        
        select.form-control {
            background-color: #333333;
            color: white;
        }
        
        .alert-warning {
            background-color: #f39c12;
            border-color: #e67e22;
            color: #1a252f;
        }
    </style>
</head>
<body>
    <!-- Dashboard Header -->
    <div class="container mt-4">
        <div class="dashboard-header">
            <div class="container">
                <h1 class="text-center">
                    <i class="icon icon-bullhorn"></i> Ads Management
                </h1>
            </div>
        </div>

        <!-- Text Controls Section -->
        <div class="text-controls-panel">
            <h2 class="mb-4">Text Controls</h2>
            
            <div class="row">
                <!-- Top Text Control -->
                <div class="col-md-6">
                    <div class="control-box">
                        <h4>Top Banner Text</h4>
                        <form method="post">
                            <div class="form-group">
                                <input class="form-control" name="text1" type="text" value="<?= htmlspecialchars($rowU1['text']) ?>"/>
                            </div>
                            <div class="form-group text-center">
                                <button class="btn btn-custom-darkblue" name="update_ads" type="submit">
                                    <i class="icon icon-check"></i> Update Text
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
                
                <!-- Ads Text Control -->
                <div class="col-md-6">
                    <div class="control-box">
                        <h4>Ads Text</h4>
                        <form method="post">
                            <div class="form-group">
                                <input class="form-control" name="text2" type="text" value="<?= htmlspecialchars($rowU2['text']) ?>"/>
                            </div>
                            <div class="form-group text-center">
                                <button class="btn btn-custom-darkblue" name="update_ads2" type="submit">
                                    <i class="icon icon-check"></i> Update Ads Text
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
        </div>
        
        <div class="row">
            <div class="col-md-6">
                <!-- API Settings Panel -->
                <div class="api-settings-panel card">
                    <div class="card-header">
                        <h3 class="m-0">
                            <i class="icon icon-cogs"></i> TMDB API Configuration
                        </h3>
                    </div>
                    <div class="card-body">
                        <form method="post">
                            <div class="form-group">
                                <label for="tmdb_api_enabled">API Status:</label>
                                <select class="form-control" id="tmdb_api_enabled" name="tmdb_api_enabled" <?= $manualImagesExist ? 'disabled' : '' ?>>
                                    <option value="1" <?= $tmdbApiState ? 'selected' : '' ?>>Enabled</option>
                                    <option value="0" <?= !$tmdbApiState ? 'selected' : '' ?>>Disabled</option>
                                </select>
                                
                                <?php if ($manualImagesExist): ?>
                                <div class="alert alert-warning mt-3">
                                    <i class="icon icon-warning-sign"></i> TMDB API is automatically disabled when manual images exist.
                                </div>
                                <?php endif; ?>
                            </div>
                            <div class="form-group text-center">
                                <button class="btn btn-custom-blue" name="toggle_tmdb" type="submit" <?= $manualImagesExist ? 'disabled' : '' ?>>
                                    <i class="icon icon-refresh"></i> Update TMDB API State
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
                
                <!-- URL Manager Panel -->
                <div class="url-manager-panel card">
                    <div class="card-header">
                        <h3 class="m-0">
                            <i class="icon icon-link"></i> Add New URL
                        </h3>
                    </div>
                    <div class="card-body">
                        <form method="post">
                            <div class="form-group">
                                <label for="url">Image URL:</label>
                                <input class="form-control" id="url" name="url" type="text" placeholder="Enter URL" required/>
                            </div>
                            <div class="form-group text-center">
                                <button class="btn btn-custom-blue" name="add_url" type="submit">
                                    <i class="icon icon-plus"></i> Add URL
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
            
            <div class="col-md-6">
                <!-- Image Gallery -->
                <div class="gallery-container card">
                    <div class="card-header bg-primary">
                        <h3 class="m-0">
                            <i class="icon icon-picture"></i> Image Gallery
                        </h3>
                    </div>
                    <div class="card-body">
                        <?php if (count($ads2Urls) > 0): ?>
                            <div class="image-showcase">
                                <?php foreach ($ads2Urls as $url): ?>
                                    <div class="image-item">
                                        <img src="<?= htmlspecialchars($url['url']) ?>" alt="Image">
                                        <div class="delete-overlay">
                                            <form method="post">
                                                <input type="hidden" name="id" value="<?= $url['id'] ?>">
                                                <button class="btn btn-danger btn-sm" name="delete_url" type="submit">
                                                    Delete
                                                </button>
                                            </form>
                                        </div>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                        <?php else: ?>
                            <div class="alert alert-info">
                                <i class="icon icon-info-sign"></i> No images have been added yet.
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://code.jquery.com/jquery-3.5.1.slim.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/@popperjs/core@2.9.2/dist/umd/popper.min.js"></script>
    <script src="https://maxcdn.bootstrapcdn.com/bootstrap/4.5.2/js/bootstrap.min.js"></script>
</body>
</html>

<?php include('includes/footer.php'); ?>